
#ifndef CRYS_HOST_TST_ECC
#define CRYS_HOST_TST_ECC

/***************************************************************** 
*  @file CRYS_HOST_TST_ECC.h
*  @brief Functions used for HOST-SEP communication to send message from HOST to SEP
*         that contains requested API to be executed in SEP and API parameters
*         Used for SEP External application ATP testing.
*
*  @author mottig
*  @remarks Copyright (C) 2004 by Discretix Technologies Ltd.
*           All Rights reserved
*/

/************************ Includes *****************************/
#include "DX_VOS_BaseTypes.h"
#include "ExtApp_CRYS_Defines.h"
#include "CRYS.h"

#ifdef __cplusplus
extern "C"
{
#endif


/************************ Defines ******************************/



/************************ Enums ********************************/

/************************ Typedefs  ****************************/

/************************Public Variables **********************/

/*********************** Public Functions **********************/

/***********************************************************************
* Function Name:                                                           
*  SEPQA_ECC_ECDSA_NonIntegratedSignVerify         
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn_ptr   - The pointer to external public key data.
* PublKeySizeInBytes -	Size of public key data in bytes (according to EC domain),
*						it should be equal to 2*modulusSize (CRYS_ECPKI_DomainID_WMDRM10) or 
*						2*modulusSize + 1byte (other domains).
* PrivKeyIn_ptr -      Pointer to private key data. 
* PrivKeySizeInBytes - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* DataIn_ptr - input data
* DataInSize - input data size
* SignatureSize - size of the sugnature
* HashMode - hash mode
* SignatureSize - the size of signature buffer 
*				  must be not less than 2*OrderSizeInBytes.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes RSA primitive encryption/decryption process test
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_ECC_ECDSA_NonIntegratedSignVerify(
                            	DxUint32_t			      KeyGenType,
                            	CRYS_ECPKI_DomainID_t     DomainID,
                            	DxUint8_t			     *PublKeyIn_ptr,       /*in*/									
                            	DxUint32_t                PublKeySizeInBytes,  /*in*/
                            	DxUint8_t			     *PrivKeyIn_ptr,     /*in*/
                            	DxUint32_t                PrivKeySizeInBytes,/*in*/
                            	CRYS_ECPKI_HASH_OpMode_t  HashMode,
                            	DxUint8_t				 *DataIn_ptr,
                            	DxUint32_t                DataInSize,
                            	DxUint32_t                SignatureSize,
                            	char                     *FailureFunctionNameOut_ptr,
                                DxUint32_t                FuncNameMaxSize);
                                
                                
/***********************************************************************
* Function Name:                                                           
*  SEPQA_ECC_ECDSA_IntegratedSignVerify         
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn_ptr   - The pointer to external public key data.
* PublKeySizeInBytes -	Size of public key data in bytes (according to EC domain),
*						it should be equal to 2*modulusSize (CRYS_ECPKI_DomainID_WMDRM10) or 
*						2*modulusSize + 1byte (other domains).
* PrivKeyIn_ptr -      Pointer to private key data. 
* PrivKeySizeInBytes - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* DataIn_ptr - input data
* DataInSize - input data size
* SignatureSize - size of the sugnature
* HashMode - hash mode
* SignatureSize - the size of signature buffer 
*				  must be not less than 2*OrderSizeInBytes.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes Acceptance test for ECC ESDSA Sign & Verify
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
**************************************************************************************/                                
CRYSError_t SEPQA_ECC_ECDSA_IntegratedSignVerify(
					DxUint32_t			      KeyGenType,
					CRYS_ECPKI_DomainID_t     DomainID,
					DxUint8_t			     *PublKeyIn_ptr,       /*in*/									
					DxUint32_t                PublKeySizeInBytes,  /*in*/
					DxUint8_t			     *PrivKeyIn_ptr,     /*in*/
					DxUint32_t                PrivKeySizeInBytes,/*in*/
					CRYS_ECPKI_HASH_OpMode_t  HashMode,
					DxUint8_t				 *DataIn_ptr,
					DxUint16_t                DataInSize,
					DxUint32_t                SignatureSize,
					char                     *FailureFunctionNameOut_ptr,
					DxUint32_t                FuncNameMaxSize);
                                
                                
/***********************************************************************
* Function Name:                                                           
*  SEPQA_ECC_DH         
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn1_ptr   - The pointer to external public key data for user1.
* PublKeySizeInBytes1 -	Size of public key data in bytes 
* PublKeyIn2_ptr   - The pointer to external public key data for user2.
* PublKeySizeInBytes2 -	Size of public key data in bytes 
* PrivKeyIn1_ptr -      Pointer to private key data for user1. 
* PrivKeySizeInBytes1 - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* PrivKeyIn2_ptr -      Pointer to private key data for user2. 
* PrivKeySizeInBytes2 - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* MaxSecretKeySize    = maximal size of the shared secret key
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes Diffie-Hellman key shared secret generation
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
**************************************************************************************/                                
CRYSError_t SEPQA_ECC_DH(
				DxUint32_t			      KeyGenType,
				CRYS_ECPKI_DomainID_t     DomainID,
				DxUint8_t			     *PublKeyIn1_ptr,       /*in*/									
				DxUint32_t                PublKeySizeInBytes1,  /*in*/
				DxUint8_t			     *PublKeyIn2_ptr,       /*in*/									
				DxUint32_t                PublKeySizeInBytes2,  /*in*/
				DxUint8_t			     *PrivKeyIn1_ptr,     /*in*/
				DxUint32_t                PrivKeySizeInBytes1,/*in*/
				DxUint8_t			     *PrivKeyIn2_ptr,     /*in*/
				DxUint32_t                PrivKeySizeInBytes2,/*in*/
				DxUint32_t                MaxSecretKeySize,
				char                     *FailureFunctionNameOut_ptr,
				DxUint32_t                FuncNameMaxSize);
				
/***********************************************************************
* Function Name:                                                           
*  SEPQA_ECC_ElGamal_EncDec         
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn_ptr   - The pointer to external public key data.
* PublKeySizeInBytes -	Size of public key data in bytes (according to EC domain),
*						it should be equal to 2*modulusSize (CRYS_ECPKI_DomainID_WMDRM10) or 
*						2*modulusSize + 1byte (other domains).
* PrivKeyIn_ptr -      Pointer to private key data. 
* PrivKeySizeInBytes - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* DataIn_ptr - input data
* DataInSize - input data size
* EncryptedMsgSize - size of the encrypted message
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
* FuncNameMaxSize - function name size
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes Diffie-Hellman key shared secret generation
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
**************************************************************************************/                                
CRYSError_t SEPQA_ECC_ElGamal_EncDec(
						 DxUint32_t			      KeyGenType,
						 CRYS_ECPKI_DomainID_t    DomainID,
						 DxUint8_t			     *PublKeyIn_ptr,       
						 DxUint32_t               PublKeySizeInBytes,  
						 DxUint8_t			     *PrivKeyIn_ptr,     
						 DxUint32_t               PrivKeySizeInBytes,
					     DxUint8_t				 *DataIn_ptr,
						 DxUint16_t               DataInSize,
						 DxUint32_t               EncryptedMsgSize,
						 char                    *FailureFunctionNameOut_ptr,
				         DxUint32_t               FuncNameMaxSize);
				         
/***********************************************************************
* Function Name:                                                           
*  SEPQA_ECC_ImportExportKeyPair         
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn_ptr   - The pointer to external public key data.
* PublKeySizeInBytes -	Size of public key data in bytes (according to EC domain),
*						it should be equal to 2*modulusSize (CRYS_ECPKI_DomainID_WMDRM10) or 
*						2*modulusSize + 1byte (other domains).
* PrivKeyIn_ptr -      Pointer to private key data. 
* PrivKeySizeInBytes - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* DataIn_ptr - input data
* DataInSize - input data size
* SignatureSize - size of the sugnature
* HashMode - hash mode
* SignatureSize - the size of signature buffer 
*				  must be not less than 2*OrderSizeInBytes.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes Acceptance test import/export of the keys
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_ECC_ImportExportKeyPair(
                            	DxUint32_t			      KeyGenType,
                            	CRYS_ECPKI_DomainID_t     DomainID,
                            	DxUint8_t			     *PublKeyIn_ptr,       /*in*/									
                            	DxUint32_t                PublKeySizeInBytes,  /*in*/
                            	DxUint8_t			     *PrivKeyIn_ptr,     /*in*/
                            	DxUint32_t                PrivKeySizeInBytes,/*in*/
                            	CRYS_ECPKI_HASH_OpMode_t  HashMode,
                            	DxUint8_t				 *DataIn_ptr,
                            	DxUint32_t                DataInSize,
                            	DxUint32_t                SignatureSize,
                            	char                     *FailureFunctionNameOut_ptr,
                                DxUint32_t                FuncNameMaxSize);
				         
				

        						    
							      
							      
							      
							    
							    
							    
									   
        						  								   
		
#ifdef __cplusplus
}
#endif

#endif  /* CRYS_HOST_TST_ECC */
